# Install (once) if needed:
# install.packages(c("sf", "ggplot2", "rnaturalearth", "rnaturalearthdata"))

library(sf)
library(ggplot2)
library(rnaturalearth)
library(rnaturalearthdata)

# 1) Get country polygons (Natural Earth, ~1:50m scale)
world <- ne_countries(scale = "medium", returnclass = "sf")

# 2) Define geographic extent covering Europe + N. Africa (lon/lat, EPSG:4326)
#    Adjust bounds if you want more/less area.
bbox_lonlat <- st_as_sfc(st_bbox(c(
  xmin = -25,  # Azores to the west
  xmax =  60,  # Ural foothills-ish to the east
  ymin =  10,  # Sahara north (~10°N) to include Maghreb/Egypt
  ymax =  72   # Fennoscandia
), crs = 4326))

# 3) Crop world to the extent (still in lon/lat)
euro_na <- st_intersection(world, bbox_lonlat)

# 4) Choose a nice equal-area projection for Europe+N. Africa (Lambert Azimuthal)
laea_eu <- "+proj=laea +lat_0=45 +lon_0=15 +datum=WGS84 +units=m +no_defs"

# 5) Project and (optionally) buffer to fix tiny topology glitches before plotting
euro_na_laea <- st_make_valid(st_transform(euro_na, laea_eu))

# 6) Plot: blank gray land, no borders or faint borders—your choice
p <- ggplot() +
  geom_sf(data = euro_na_laea, fill = "grey85", color = "grey60", linewidth = 0.2) +
  coord_sf(crs = laea_eu, expand = FALSE) +
  theme_void() +
  theme(
    plot.background = element_rect(fill = "white", color = NA),
    panel.background = element_rect(fill = "white", color = NA)
  )

# 7) Show it
print(p)

# 8) Save high-res outputs (uncomment what you need)
# ggsave("europe_n_africa_gray.png", p, width = 8, height = 7, dpi = 300)
# ggsave("europe_n_africa_gray.svg", p, width = 8, height = 7)
